<?php

namespace App\Core;

use App\Models\VpsPlan;
use App\Models\User;

class DigitalOceanClient
{
    private string $apiToken;
    private string $region;
    private string $image;
    private array $sshKeys;
    private string $tag;

    public function __construct()
    {
        $config = require dirname(__DIR__, 2) . '/config/digitalocean.php';
        $this->apiToken = $config['api_token'] ?? '';
        $this->region = $config['region'] ?? 'sgp1';
        $this->image = $config['image'] ?? 'ubuntu-22-04-x64';
        $this->sshKeys = $config['ssh_keys'] ?? [];
        $this->tag = $config['tag'] ?? 'sherifcloud-vps';
    }

    public function createDroplet(User $user, VpsPlan $plan, string $name, ?string $imageSlug = null): ?array
    {
        if ($this->apiToken === '' || $this->apiToken === 'YOUR_DO_API_TOKEN') {
            return null;
        }

        $url = 'https://api.digitalocean.com/v2/droplets';

        $payload = [
            'name' => $name,
            'region' => $this->region,
            'size' => $plan->size_slug,
            'image' => $imageSlug ?: $this->image,
            'backups' => false,
            'ipv6' => true,
            'monitoring' => true,
            'ssh_keys' => $this->sshKeys,
            'tags' => [$this->tag, 'user-' . $user->id],
            'user_data' => null
        ];

        $ch = curl_init();
        $headers = [
            'Authorization: Bearer ' . $this->apiToken,
            'Content-Type: application/json',
            'Accept: application/json'
        ];

        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($payload)
        ];

        curl_setopt_array($ch, $options);
        $response = curl_exec($ch);

        if ($response === false) {
            $error = curl_error($ch);
            error_log('DigitalOcean API request error: ' . $error);
            curl_close($ch);
            return null;
        }

        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($status >= 200 && $status < 300) {
            $decoded = json_decode($response, true);
            return $decoded;
        }

        error_log('DigitalOcean API non-2xx response: HTTP ' . $status . ' for create droplet');
        return null;
    }

    public function getDroplet(int $dropletId): ?array
    {
        if ($this->apiToken === '' || $this->apiToken === 'YOUR_DO_API_TOKEN') {
            return null;
        }

        $url = 'https://api.digitalocean.com/v2/droplets/' . $dropletId;

        $ch = curl_init();
        $headers = [
            'Authorization: Bearer ' . $this->apiToken,
            'Accept: application/json'
        ];

        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers
        ];

        curl_setopt_array($ch, $options);
        $response = curl_exec($ch);

        if ($response === false) {
            $error = curl_error($ch);
            error_log('DigitalOcean API request error: ' . $error);
            curl_close($ch);
            return null;
        }

        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($status >= 200 && $status < 300) {
            $decoded = json_decode($response, true);
            return $decoded;
        }

        error_log('DigitalOcean API non-2xx response: HTTP ' . $status . ' for get droplet');
        return null;
    }

    public function countDroplets(): ?int
    {
        if ($this->apiToken === '' || $this->apiToken === 'YOUR_DO_API_TOKEN') {
            return null;
        }

        $url = 'https://api.digitalocean.com/v2/droplets?per_page=1';

        $ch = curl_init();
        $headers = [
            'Authorization: Bearer ' . $this->apiToken,
            'Accept: application/json'
        ];

        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers
        ];

        curl_setopt_array($ch, $options);
        $response = curl_exec($ch);

        if ($response === false) {
            $error = curl_error($ch);
            error_log('DigitalOcean API request error: ' . $error);
            curl_close($ch);
            return null;
        }

        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($status >= 200 && $status < 300) {
            $decoded = json_decode($response, true);
            if (isset($decoded['meta']['total'])) {
                return (int) $decoded['meta']['total'];
            }
            if (isset($decoded['droplets']) && is_array($decoded['droplets'])) {
                return count($decoded['droplets']);
            }
        }

        error_log('DigitalOcean API non-2xx response: HTTP ' . $status . ' for list droplets');
        return null;
    }
}
