<?php

namespace App\Core;

class PterodactylClient
{
    private string $panelUrl;
    private string $applicationKey;
    private string $clientKey;
    private string $applicationPath;
    private string $clientPath;

    public function __construct()
    {
        $config = require dirname(__DIR__, 2) . '/config/pterodactyl.php';
        $this->panelUrl = rtrim($config['panel_url'], '/');
        $this->applicationKey = $config['application_api_key'];
        $this->clientKey = $config['client_api_key'];
        $this->applicationPath = $config['application_api_path'];
        $this->clientPath = $config['client_api_path'];
    }

    public function createUser(string $email, string $username, string $firstName, string $lastName, string $password): ?array
    {
        $payload = [
            'email' => $email,
            'username' => $username,
            'first_name' => $firstName,
            'last_name' => $lastName,
            'password' => $password
        ];

        return $this->requestApplication('POST', '/users', $payload);
    }

    public function findUserByEmail(string $email): ?array
    {
        $query = '/users?filter[email]=' . urlencode($email);
        $response = $this->requestApplication('GET', $query);
        if (!$response || !isset($response['data']) || !is_array($response['data']) || count($response['data']) === 0) {
            return null;
        }
        if (!isset($response['data'][0]['attributes']) || !is_array($response['data'][0]['attributes'])) {
            return null;
        }
        return $response['data'][0]['attributes'];
    }

    public function createServer(int $userId, string $name, int $memory, int $disk): ?array
    {
        $config = require dirname(__DIR__, 2) . '/config/pterodactyl.php';

        $payload = [
            'name' => $name,
            'user' => $userId,
            'egg' => $config['default_egg_id'],
            'docker_image' => 'ghcr.io/parkervcp/yolks:nodejs_18',
            'startup' => 'npm start',
            'limits' => [
                'memory' => $memory,
                'disk' => $disk,
                'cpu' => 0,
                'swap' => 0,
                'io' => 500
            ],
            'environment' => [],
            'feature_limits' => [
                'databases' => 0,
                'backups' => 0,
                'allocations' => 1
            ],
            'allocation' => [
                'default' => $config['default_allocation_id']
            ]
        ];

        return $this->requestApplication('POST', '/servers', $payload);
    }

    public function powerAction(string $serverIdentifier, string $signal): ?array
    {
        $payload = [
            'signal' => $signal
        ];

        return $this->requestClient('POST', '/servers/' . $serverIdentifier . '/power', $payload);
    }

    public function getServerResources(string $serverIdentifier): ?array
    {
        return $this->requestClient('GET', '/servers/' . $serverIdentifier . '/resources');
    }

    public function suspendServer(int $serverId): ?array
    {
        return $this->requestApplication('POST', '/servers/' . $serverId . '/suspend');
    }

    public function unsuspendServer(int $serverId): ?array
    {
        return $this->requestApplication('POST', '/servers/' . $serverId . '/unsuspend');
    }

    public function deleteServer(int $serverId): ?array
    {
        return $this->requestApplication('DELETE', '/servers/' . $serverId);
    }

    private function requestApplication(string $method, string $endpoint, array $data = []): ?array
    {
        return $this->request($method, $this->applicationPath, $this->applicationKey, $endpoint, $data);
    }

    private function requestClient(string $method, string $endpoint, array $data = []): ?array
    {
        return $this->request($method, $this->clientPath, $this->clientKey, $endpoint, $data);
    }

    private function request(string $method, string $path, string $token, string $endpoint, array $data = []): ?array
    {
        if ($token === '' || $token === 'YOUR_APPLICATION_API_KEY' || $token === 'YOUR_CLIENT_API_KEY') {
            return null;
        }

        $url = $this->panelUrl . $path . $endpoint;

        $ch = curl_init();
        $headers = [
            'Authorization: Bearer ' . $token,
            'Accept: application/json',
            'Content-Type: application/json'
        ];

        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_CUSTOMREQUEST => $method
        ];

        if ($method !== 'GET' && !empty($data)) {
            $options[CURLOPT_POSTFIELDS] = json_encode($data);
        }

        curl_setopt_array($ch, $options);
        $response = curl_exec($ch);

        if ($response === false) {
            $error = curl_error($ch);
            error_log('Pterodactyl API request error: ' . $error);
            curl_close($ch);
            return null;
        }

        $statusCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($statusCode >= 200 && $statusCode < 300) {
            $decoded = json_decode($response, true);
            return $decoded;
        }

        error_log('Pterodactyl API non-2xx response: HTTP ' . $statusCode . ' for ' . $path . $endpoint);

        return null;
    }
}
