<?php

namespace App\Models;

use App\Core\Database;
use PDO;

class Server
{
    public int $id;
    public int $user_id;
    public int $package_id;
    public string $name;
    public ?int $pterodactyl_id;
    public ?string $identifier;
    public string $status;

    public static function userHasFreeServer(int $userId): bool
    {
        $pdo = Database::getConnection();
        $sql = 'SELECT COUNT(*) AS c
                FROM servers s
                JOIN packages p ON p.id = s.package_id
                WHERE s.user_id = :user_id AND p.is_free = 1';
        $stmt = $pdo->prepare($sql);
        $stmt->execute(['user_id' => $userId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) {
            return false;
        }
        return (int) $row['c'] > 0;
    }

    public static function forUser(int $userId): array
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('SELECT * FROM servers WHERE user_id = :user_id ORDER BY id DESC');
        $stmt->execute(['user_id' => $userId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        return array_map([self::class, 'fromArray'], $rows);
    }

    public static function all(): array
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->query('SELECT * FROM servers ORDER BY id DESC');
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        return array_map([self::class, 'fromArray'], $rows);
    }

    public static function findById(int $id): ?self
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('SELECT * FROM servers WHERE id = :id LIMIT 1');
        $stmt->execute(['id' => $id]);
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$data) {
            return null;
        }
        return self::fromArray($data);
    }

    public static function create(int $userId, int $packageId, string $name): self
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('INSERT INTO servers (user_id, package_id, name, status) VALUES (:user_id, :package_id, :name, :status)');
        $stmt->execute([
            'user_id' => $userId,
            'package_id' => $packageId,
            'name' => $name,
            'status' => 'deploying'
        ]);
        $id = (int) $pdo->lastInsertId();
        return self::findById($id);
    }

    public static function createWithPterodactyl(int $userId, int $packageId, string $name, int $pterodactylId, string $identifier, string $status = 'online'): self
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('INSERT INTO servers (user_id, package_id, name, pterodactyl_id, identifier, status) VALUES (:user_id, :package_id, :name, :pterodactyl_id, :identifier, :status)');
        $stmt->execute([
            'user_id' => $userId,
            'package_id' => $packageId,
            'name' => $name,
            'pterodactyl_id' => $pterodactylId,
            'identifier' => $identifier,
            'status' => $status
        ]);
        $id = (int) $pdo->lastInsertId();
        return self::findById($id);
    }

    public static function updatePterodactyl(int $id, int $pterodactylId, string $identifier): void
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('UPDATE servers SET pterodactyl_id = :pid, identifier = :identifier WHERE id = :id');
        $stmt->execute([
            'pid' => $pterodactylId,
            'identifier' => $identifier,
            'id' => $id
        ]);
    }

    public static function updateStatus(int $id, string $status): void
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('UPDATE servers SET status = :status WHERE id = :id');
        $stmt->execute([
            'status' => $status,
            'id' => $id
        ]);
    }

    public static function deleteById(int $id): void
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('DELETE FROM servers WHERE id = :id');
        $stmt->execute(['id' => $id]);
    }

    public static function fromArray(array $data): self
    {
        $server = new self();
        $server->id = (int) $data['id'];
        $server->user_id = (int) $data['user_id'];
        $server->package_id = (int) $data['package_id'];
        $server->name = $data['name'];
        $server->pterodactyl_id = isset($data['pterodactyl_id']) ? (int) $data['pterodactyl_id'] : null;
        $server->identifier = $data['identifier'] ?? null;
        $server->status = $data['status'];
        return $server;
    }
}
