document.addEventListener("DOMContentLoaded", function () {
    var navToggle = document.getElementById("nav-toggle");
    var mobileMenu = document.getElementById("mobile-menu");
    if (navToggle && mobileMenu) {
        navToggle.addEventListener("click", function () {
            if (mobileMenu.classList.contains("hidden")) {
                mobileMenu.classList.remove("hidden");
            } else {
                mobileMenu.classList.add("hidden");
            }
        });
    }

    var typingElements = document.querySelectorAll(".terminal-typing");
    typingElements.forEach(function (el) {
        var linesRaw = el.getAttribute("data-lines");
        if (!linesRaw) {
            return;
        }
        var lines;
        try {
            lines = JSON.parse(linesRaw);
        } catch (e) {
            lines = [];
        }
        var currentLine = 0;
        var currentChar = 0;
        function typeNext() {
            if (!lines.length) {
                return;
            }
            if (currentChar <= lines[currentLine].length) {
                el.textContent = lines[currentLine].slice(0, currentChar);
                currentChar += 1;
                setTimeout(typeNext, 40);
            } else {
                currentLine = (currentLine + 1) % lines.length;
                currentChar = 0;
                setTimeout(typeNext, 600);
            }
        }
        typeNext();
    });

    var logContainer = document.querySelector(".live-log-body");
    if (logContainer) {
        var fakeStates = [
            "[node-01] Health check OK. All containers healthy.",
            "[node-02] Deploying new bot instance...",
            "[scheduler] Tick cycle completed in 32ms.",
            "[ptero-api] Metrics synced.",
            "[alert] No incidents reported in the last 24h."
        ];
        var idx = 0;
        function pushLog() {
            var now = new Date();
            var line = document.createElement("div");
            var time = now.toISOString().split("T")[1].split(".")[0];
            line.textContent = "[" + time + "] " + fakeStates[idx % fakeStates.length];
            logContainer.appendChild(line);
            logContainer.scrollTop = logContainer.scrollHeight;
            idx += 1;
            setTimeout(pushLog, 1800);
        }
        pushLog();
    }

    var orderButtons = document.querySelectorAll(".order-button");
    orderButtons.forEach(function (btn) {
        btn.addEventListener("click", function () {
            btn.classList.add("loading");
        });
    });

    var serverCards = document.querySelectorAll(".server-card");
    if (serverCards.length > 0 && window.SHERIF_CLOUD_BASE_URL) {
        var notifyContainer = document.createElement("div");
        notifyContainer.className = "fixed bottom-4 right-4 z-50 space-y-2 text-xs";
        document.body.appendChild(notifyContainer);

        function pushNotification(message) {
            var box = document.createElement("div");
            box.className = "bg-red-500/90 text-white px-3 py-2 rounded-lg shadow-lg";
            box.textContent = message;
            notifyContainer.appendChild(box);
            setTimeout(function () {
                box.remove();
            }, 5000);
        }

        function formatBytes(bytes) {
            if (!bytes || bytes <= 0) return "0 MB";
            var mb = bytes / (1024 * 1024);
            return mb.toFixed(1) + " MB";
        }

        function pollServers() {
            serverCards.forEach(function (card) {
                var id = card.getAttribute("data-server-id");
                if (!id) return;
                var url = window.SHERIF_CLOUD_BASE_URL + "/api/server/status?id=" + encodeURIComponent(id);
                fetch(url, { credentials: "same-origin" })
                    .then(function (res) {
                        return res.json();
                    })
                    .then(function (data) {
                        if (!data || data.error) return;
                        var statusEl = card.querySelector(".server-status");
                        var cpuEl = card.querySelector(".server-cpu");
                        var ramEl = card.querySelector(".server-ram");
                        var diskEl = card.querySelector(".server-disk");
                        if (statusEl) {
                            statusEl.textContent = data.status || "unknown";
                            statusEl.classList.remove("text-emerald-400", "text-red-400", "text-slate-400");
                            if (data.status === "running" || data.status === "online") {
                                statusEl.classList.add("text-emerald-400");
                            } else if (data.status === "offline" || data.status === "stopping" || data.status === "stopped") {
                                statusEl.classList.add("text-red-400");
                                pushNotification("Server #" + id + " saat ini offline.");
                            } else {
                                statusEl.classList.add("text-slate-400");
                            }
                        }
                        if (cpuEl && typeof data.cpu === "number") {
                            cpuEl.textContent = data.cpu.toFixed(1) + "%";
                        }
                        if (ramEl) {
                            ramEl.textContent = formatBytes(data.memory);
                        }
                        if (diskEl) {
                            diskEl.textContent = formatBytes(data.disk);
                        }
                    })
                    .catch(function () {
                    });
            });
        }

        pollServers();
        setInterval(pollServers, 5000);

        var powerButtons = document.querySelectorAll(".power-action");
        powerButtons.forEach(function (btn) {
            btn.addEventListener("click", function () {
                var card = btn.closest(".server-card");
                if (!card) return;
                var id = card.getAttribute("data-server-id");
                var signal = btn.getAttribute("data-signal");
                if (!id || !signal) return;
                var formData = new FormData();
                formData.append("id", id);
                formData.append("signal", signal);
                var url = window.SHERIF_CLOUD_BASE_URL + "/api/server/power";
                btn.disabled = true;
                fetch(url, {
                    method: "POST",
                    body: formData,
                    credentials: "same-origin"
                })
                    .then(function (res) {
                        return res.json();
                    })
                    .then(function (data) {
                        if (data && data.error) {
                            pushNotification("Gagal mengirim perintah " + signal + " ke server #" + id + ".");
                        }
                    })
                    .catch(function () {
                        pushNotification("Gagal menghubungi API untuk server #" + id + ".");
                    })
                    .finally(function () {
                        btn.disabled = false;
                    });
            });
        });
    }
});
